package com.sample;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Random;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.drools.KnowledgeBase;
import org.drools.KnowledgeBaseFactory;
import org.drools.builder.KnowledgeBuilder;
import org.drools.builder.KnowledgeBuilderError;
import org.drools.builder.KnowledgeBuilderErrors;
import org.drools.builder.KnowledgeBuilderFactory;
import org.drools.builder.ResourceType;
import org.drools.RuleBaseConfiguration;
import org.drools.io.ResourceFactory;
import org.drools.logger.KnowledgeRuntimeLogger;
import org.drools.logger.KnowledgeRuntimeLoggerFactory;
import org.drools.runtime.StatelessKnowledgeSession;

/**
 * This is a sample class to launch a rule.
 */
public class DroolsTest {

	private static final int THREADS = 5;
	private static final int CALLS = 10;
	
	private static final long BASE_NUMBER = (long)Math.pow(10, 12);
	private static final long MAX_NUMBER =  (long)Math.pow(10, 13)-1;
	private static final long NUMBER_SPREAD = MAX_NUMBER - BASE_NUMBER;
	
	private static final boolean RANDOM_RULES = true;
	
	private static final ExecutorService SERVICE = Executors.newFixedThreadPool(THREADS);
	
	public static final void main(String[] args) {
		
		final int numRules = Integer.parseInt(args[0]);
		final int numPhoneNumbers = Integer.parseInt(args[1]);
		
		try {
			
			final CountDownLatch shutdownLatch = new CountDownLatch(CALLS);
			
			// load up the knowledge base
			final KnowledgeBase kbase = readKnowledgeBase(numRules, RANDOM_RULES);
			
			for(int i = 0; i < CALLS; i++) {
				SERVICE.execute(new Runnable() {
					public void run() {
						StatelessKnowledgeSession ksession = kbase.newStatelessKnowledgeSession();
						long time = System.currentTimeMillis();
						ksession.execute(getRandomInput(numPhoneNumbers));
						long elapsed = System.currentTimeMillis() - time;
						System.out.println("Rules execution time: " + elapsed + "ms, per number: " + ((double)elapsed / numPhoneNumbers) + "ms");
						shutdownLatch.countDown();
					}
				});
			}
			
			shutdownLatch.await();
			SERVICE.shutdown();
			System.out.println("DONE!");
			
		} catch (Throwable t) {
			t.printStackTrace();
		}
	}

	private static final Iterable<PhoneNumber> getRandomInput(int numPhoneNumbers) {
		Random rnd = new Random();
		List<PhoneNumber> numbers = new ArrayList<PhoneNumber>(numPhoneNumbers);
		for(int i = 0; i < numPhoneNumbers; i++) {
			numbers.add(new PhoneNumber(Long.toString(BASE_NUMBER + Math.abs(rnd.nextLong())%NUMBER_SPREAD)));
		}
		return numbers;
	}
	
	private static KnowledgeBase readKnowledgeBase(final int numRules, final boolean random) throws Exception {

		long time = System.currentTimeMillis();
		
		KnowledgeBuilder kbuilder = KnowledgeBuilderFactory.newKnowledgeBuilder();
		kbuilder.add(ResourceFactory.newReaderResource(new StringReader(getRules(numRules, random))), ResourceType.DRL);
		KnowledgeBuilderErrors errors = kbuilder.getErrors();
		if (errors.size() > 0) {
			for (KnowledgeBuilderError error: errors) {
				System.err.println(error);
			}
			throw new IllegalArgumentException("Could not parse knowledge.");
		}
		
		RuleBaseConfiguration rbc = new RuleBaseConfiguration();
		rbc.setSequential(true);
		
		KnowledgeBase kbase = KnowledgeBaseFactory.newKnowledgeBase(rbc);
		
		kbase.addKnowledgePackages(kbuilder.getKnowledgePackages());
		
		System.out.println("Rules load time: " + (System.currentTimeMillis() - time) + "ms" );
		
		return kbase;
	}
	
	private static final String getRules(final int numRules, final boolean random) {
		
		Random rnd = new Random();
		
		StringBuilder rules = new StringBuilder();
		rules.append("package com.sample\n");
		rules.append("import com.sample.DroolsTest.PhoneNumber;\n");
		
		for(int i = 0; i < numRules; i++) {
			
			String number = null;
			
			if(random) {
				number = Long.toString(BASE_NUMBER + Math.abs(rnd.nextLong())%NUMBER_SPREAD);
			} else {
				number = Long.toString(BASE_NUMBER + i);
			}
			rules.append("rule \"PhoneNumber");
			rules.append(number);
			rules.append("\"\n");
			
			rules.append("when\n");
			rules.append("p : PhoneNumber( ");
			for(int j = 0; j < 13; j++) {
				rules.append("digit");
				rules.append(j);
				rules.append(" == '");
				rules.append(number.charAt(j));
				rules.append("',");
			}
			rules.deleteCharAt(rules.length()-1);
			rules.append(")\n");
			rules.append("then\n");
			rules.append("p.setPrice(");
			rules.append(rnd.nextDouble());
			rules.append(");\n");
			rules.append("end\n");
		}
		
		return rules.toString();
	}

	public static class PhoneNumber {

		private final String number;
		private final char digit0;
		private final char digit1;
		private final char digit2;
		private final char digit3;
		private final char digit4;
		private final char digit5;
		private final char digit6;
		private final char digit7;
		private final char digit8;
		private final char digit9;
		private final char digit10;
		private final char digit11;
		private final char digit12;
		
		private double price;

		public PhoneNumber(String number) {
			this.number = number;
			digit0 = number.charAt(0);
			digit1 = number.charAt(1);
			digit2 = number.charAt(2);
			digit3 = number.charAt(3);
			digit4 = number.charAt(4);
			digit5 = number.charAt(5);
			digit6 = number.charAt(6);
			digit7 = number.charAt(7);
			digit8 = number.charAt(8);
			digit9 = number.charAt(9);
			digit10 = number.charAt(10);
			digit11 = number.charAt(11);
			digit12 = number.charAt(12);
		}
		
		public char getDigit0() {
			return digit0;
		}
		public char getDigit1() {
			return digit1;
		}
		public char getDigit2() {
			return digit2;
		}
		public char getDigit3() {
			return digit3;
		}
		public char getDigit4() {
			return digit4;
		}
		public char getDigit5() {
			return digit5;
		}
		public char getDigit6() {
			return digit6;
		}
		public char getDigit7() {
			return digit7;
		}
		public char getDigit8() {
			return digit8;
		}
		public char getDigit9() {
			return digit9;
		}
		public char getDigit10() {
			return digit10;
		}
		public char getDigit11() {
			return digit11;
		}
		public char getDigit12() {
			return digit12;
		}
		public double getPrice() {
			return price;
		}
		public void setPrice(double price) {
			this.price = price;
		}	
		
		public String toString() {
			return number;
		}
	}

}